<?php

// Load environment configuration
require_once dirname(__FILE__) . '/../constants.php';

class PaymentGatewayFactory
{
  public static function create($provider, $config)
  {
    switch ($provider) {
      case 'midtrans':
        return new MidtransPayment($config);
      case 'tripay':
        return new TripayPayment($config);
      case 'xendit':
        return new XenditPayment($config);
      default:
        throw new Exception("Unsupported payment provider: $provider");
    }
  }
}

interface PaymentGatewayInterface
{
  public function createTransaction($params);
  public function handleNotification($request);
  public function getRedirectUrl($transactionData);
}

class MidtransPayment implements PaymentGatewayInterface
{
  private $clientKey;
  private $serverKey;

  public function __construct($config)
  {
    $this->clientKey = $config['tclientkey'];
    $this->serverKey = $config['tserverkey'];
    require_once dirname(__FILE__) . '/../vendor/autoload.php';
    \Midtrans\Config::$serverKey = $this->serverKey;
    \Midtrans\Config::$isProduction = (APP_ENV === 'production');
    \Midtrans\Config::$isSanitized = true;
    \Midtrans\Config::$is3ds = true;
  }

  public function createTransaction($params)
  {
    $transaction_details = array(
      'order_id' => $params['order_id'],
      'gross_amount' => $params['gross_amount'],
    );

    $item_details = array(
      'id' => $params['item_id'],
      'price' => $params['gross_amount'],
      'quantity' => 1,
      'name' => $params['item_name']
    );

    $customer_details = array(
      'first_name' => $params['customer_name'],
      'phone' => $params['customer_phone'],
    );

    $snapParams = array(
      'transaction_details' => $transaction_details,
      'customer_details' => $customer_details,
      'item_details' => array($item_details),
    );

    return \Midtrans\Snap::createTransaction($snapParams);
  }

  public function handleNotification($request)
  {
    $notif = new \Midtrans\Notification();
    return [
      'transaction_status' => $notif->transaction_status,
      'payment_type' => $notif->payment_type,
      'order_id' => $notif->order_id,
      'fraud_status' => $notif->fraud_status ?? null,
    ];
  }

  public function getRedirectUrl($transactionData)
  {
    return $transactionData->redirect_url;
  }
}

class TripayPayment implements PaymentGatewayInterface
{
  private $apiKey;
  private $privateKey;
  private $merchantCode;
  private $isProduction;

  public function __construct($config)
  {
    $this->apiKey = $config['tclientkey']; // API Key
    $this->privateKey = $config['tserverkey']; // Private Key
    $this->merchantCode = $config['merchant_code'];
    $this->isProduction = (APP_ENV === 'production');
  }

  public function createTransaction($params)
  {
    $paymentMethod = $params['payment_channel'] ?? 'BRIVA'; // Use selected channel or default to BRIVA

    $data = [
      'method' => $paymentMethod,
      'merchant_ref' => $params['order_id'],
      'amount' => $params['gross_amount'],
      'customer_name' => $params['customer_name'],
      'customer_email' => $params['customer_email'] ?? 'customer@example.com',
      'customer_phone' => $params['customer_phone'],
      'order_items' => [
        [
          'sku' => $params['item_id'],
          'name' => $params['item_name'],
          'price' => $params['gross_amount'],
          'quantity' => 1
        ]
      ],
      'return_url' => APP_ENV === 'production' ? APP_URL . '/payment/notification-handler' : 'https://webhook.site/19e74e40-682a-428d-8cdb-d57ac66e362a',
      'expired_time' => (time() + (24 * 60 * 60)), // 24 hours
      'signature' => $this->generateSignature($params['order_id'], $params['gross_amount'])
    ];

    $url = $this->isProduction ? 'https://tripay.co.id/api/transaction/create' : 'https://tripay.co.id/api-sandbox/transaction/create';

    $response = $this->makeApiCall($url, $data);
    return $response;
  }

  public function handleNotification($request)
  {
    return [
      'transaction_status' => $request['status'] ?? 'pending',
      'payment_type' => $request['payment_method'] ?? 'tripay',
      'order_id' => $request['merchant_ref'] ?? '',
      'fraud_status' => null,
    ];
  }

  public function getRedirectUrl($transactionData)
  {
    return $transactionData['data']['checkout_url'] ?? '';
  }

  private function generateSignature($merchantRef, $amount)
  {
    $payload = $this->merchantCode . $merchantRef . $amount;
    return hash_hmac('sha256', $payload, $this->privateKey);
  }

  private function makeApiCall($url, $data)
  {
    $curl = curl_init();
    curl_setopt_array($curl, [
      CURLOPT_URL => $url,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_HTTPHEADER => [
        'Authorization: Bearer ' . $this->apiKey,
        'Content-Type: application/json',
      ],
      CURLOPT_POSTFIELDS => json_encode($data),
    ]);

    $response = curl_exec($curl);
    curl_close($curl);
    return json_decode($response, true);
  }
}

class XenditPayment implements PaymentGatewayInterface
{
  private $apiKey;
  private $webhookToken;

  public function __construct($config)
  {
    $this->apiKey = $config['tclientkey']; // Xendit uses only API key
    $this->webhookToken = $config['webhook_token'];
  }

  public function createTransaction($params)
  {
    $data = [
      'external_id' => $params['order_id'],
      'amount' => $params['gross_amount'],
      'payer_email' => $params['customer_email'] ?? 'customer@example.com',
      'description' => $params['item_name'],
      'success_redirect_url' => 'https://yourdomain.com/payment/success',
      'failure_redirect_url' => 'https://yourdomain.com/payment/failed'
    ];

    $response = $this->makeApiCall('https://api.xendit.co/v2/invoices', $data);
    return $response;
  }

  public function handleNotification($request)
  {
    return [
      'transaction_status' => $this->mapXenditStatus($request['status'] ?? 'PENDING'),
      'payment_type' => 'xendit',
      'order_id' => $request['external_id'] ?? '',
      'fraud_status' => null,
    ];
  }

  public function getRedirectUrl($transactionData)
  {
    return $transactionData['invoice_url'] ?? '';
  }

  private function mapXenditStatus($xenditStatus)
  {
    $statusMap = [
      'PAID' => 'settlement',
      'PENDING' => 'pending',
      'EXPIRED' => 'expire',
      'FAILED' => 'deny'
    ];
    return $statusMap[$xenditStatus] ?? 'pending';
  }

  private function makeApiCall($url, $data)
  {
    $curl = curl_init();
    curl_setopt_array($curl, [
      CURLOPT_URL => $url,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_HTTPHEADER => [
        'Authorization: Basic ' . base64_encode($this->apiKey . ':'),
        'Content-Type: application/json',
      ],
      CURLOPT_POSTFIELDS => json_encode($data),
    ]);

    $response = curl_exec($curl);
    curl_close($curl);
    return json_decode($response, true);
  }
}
